// This file simulates the output of a build process (e.g., Webpack, Vite).
// In a real build, ALL JSX would be transpiled into React.createElement calls.
// For demonstration purposes, only the header section is manually converted.
// The rest of the App component still uses JSX, but remember it MUST be
// transpiled by a build tool for your Chrome extension to work.

// React and ReactDOM are assumed to be available globally because they are
// loaded via <script> tags in popup.html (UMD builds).
const { useState, useEffect } = React; // Destructure useState and useEffect from global React

// Main App component
const App = () => {
    // State variables for the scraping status section
    const [username, setUsername] = useState('example_user');
    const [instagramLink, setInstagramLink] = useState('https://www.instagram.com/example_user/');
    const [followerCount, setFollowerCount] = useState(12345);
    const [scrapedCount, setScrapedCount] = useState(0);
    const [isScraping, setIsScraping] = useState(false);

    // State for LLM modal
    const [showLLMModal, setShowLLMModal] = useState(false);
    const [llmModalTitle, setLlmModalTitle] = useState('');
    const [llmModalContent, setLlmModalContent] = useState('');
    const [llmLoading, setLlmLoading] = useState(false);

    // Demo data for the scraped users table
    const [scrapedUsers, setScrapedUsers] = useState([
        {
            no: 1,
            avatar: 'https://placehold.co/40x40/FF6347/FFFFFF?text=A1',
            username: 'travel_explorer',
            fullname: 'Alice Wonderland',
            userId: '123456789',
            followedByYou: true,
            isVerified: false,
        },
        {
            no: 2,
            avatar: 'https://placehold.co/40x40/4682B4/FFFFFF?text=B2',
            username: 'foodie_guru',
            fullname: 'Bob Chef',
            userId: '987654321',
            followedByYou: false,
            isVerified: true,
        },
        {
            no: 3,
            avatar: 'https://placehold.co/40x40/3CB371/FFFFFF?text=C3',
            username: 'art_lover',
            fullname: 'Charlie Painter',
            userId: '112233445',
            followedByYou: true,
            isVerified: false,
        },
        {
            no: 4,
            avatar: 'https://placehold.co/40x40/DAA520/FFFFFF?text=D4',
            username: 'fitness_freak',
            fullname: 'Diana Strong',
            userId: '556677889',
            followedByYou: false,
            isVerified: true,
        },
        {
            no: 5,
            avatar: 'https://placehold.co/40x40/9370DB/FFFFFF?text=E5',
            username: 'tech_wiz',
            fullname: 'Eve Coder',
            userId: '998877665',
            followedByYou: true,
            isVerified: false,
        },
    ]);

    // Calculate progress percentage
    const progressPercentage = followerCount > 0 ? (scrapedCount / followerCount) * 100 : 0;

    // Function to simulate scraping progress
    useEffect(() => {
        let interval;
        if (isScraping && scrapedCount < followerCount) {
            interval = setInterval(() => {
                setScrapedCount((prevCount) => {
                    const newCount = prevCount + Math.floor(Math.random() * 500) + 1; // Simulate scraping more followers
                    if (newCount >= followerCount) {
                        setIsScraping(false); // Stop scraping when target reached
                        return followerCount;
                    }
                    return newCount;
                });
            }, 500); // Update every 0.5 seconds
        } else if (scrapedCount >= followerCount) {
            setIsScraping(false);
        }
        return () => clearInterval(interval); // Cleanup interval on component unmount or if scraping stops
    }, [isScraping, scrapedCount, followerCount]);

    // Custom Message Box (replaces alert/confirm)
    const showMessageBox = (title, message) => {
        setLlmModalTitle(title);
        setLlmModalContent(message);
        setShowLLMModal(true);
    };

    // Handle download button click
    const handleDownload = () => {
        showMessageBox('Download Data', 'Download button clicked! (In a real app, this would download data)');
        // In a real application, you would convert scrapedUsers to CSV/JSON and trigger a download
    };

    // Handle stop scraping button click
    const handleStopScraping = () => {
        setIsScraping(false);
        showMessageBox('Scraping Status', 'Scraping stopped!');
    };

    // Handle start scraping (for demo purposes, just resets and starts)
    const handleStartScraping = () => {
        setScrapedCount(0); // Reset scraped count
        setIsScraping(true);
        showMessageBox('Scraping Status', 'Scraping started!');
    };

    // Function to generate a bio idea using Gemini API
    const handleGenerateBioIdea = async (user) => {
        setLlmLoading(true);
        setLlmModalTitle(`Bio Idea for ${user.username}`);
        setLlmModalContent('Generating a creative bio idea...');
        setShowLLMModal(true);

        try {
            let chatHistory = [];
            const prompt = `Generate a short, creative, and engaging Instagram bio idea for a user named "${user.fullname}" with the username "${user.username}". Focus on a niche if suggested by the username (e.g., 'travel', 'foodie', 'art', 'fitness', 'tech'). Keep it concise and use emojis.`;
            chatHistory.push({ role: "user", parts: [{ text: prompt }] });
            const payload = { contents: chatHistory };
            const apiKey = ""; // Leave as-is, Canvas will provide at runtime
            const apiUrl = `https://generativelanguage.googleapis.com/v1beta/models/gemini-2.0-flash:generateContent?key=${apiKey}`;

            const response = await fetch(apiUrl, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(payload)
            });

            const result = await response.json();

            if (result.candidates && result.candidates.length > 0 &&
                result.candidates[0].content && result.candidates[0].content.parts &&
                result.candidates[0].content.parts.length > 0) {
                const text = result.candidates[0].content.parts[0].text;
                setLlmModalContent(text);
            } else {
                setLlmModalContent('Failed to generate bio idea. Please try again.');
                console.error('Gemini API response structure unexpected:', result);
            }
        } catch (error) {
            setLlmModalContent('Error generating bio idea: ' + error.message);
            console.error('Error calling Gemini API for bio idea:', error);
        } finally {
            setLlmLoading(false);
        }
    };

    // Function to generate engagement insights using Gemini API
    const handleGenerateEngagementInsights = async () => {
        setLlmLoading(true);
        setLlmModalTitle('Engagement Insights ✨');
        setLlmModalContent('Generating insights and tips...');
        setShowLLMModal(true);

        try {
            let chatHistory = [];
            const prompt = `Given a total follower count of ${followerCount} and ${scrapedCount} followers scraped, provide some general insights on Instagram engagement. Suggest 3-5 actionable tips to improve engagement for an Instagram account.`;
            chatHistory.push({ role: "user", parts: [{ text: prompt }] });
            const payload = { contents: chatHistory };
            const apiKey = ""; // Leave as-is, Canvas will provide at runtime
            const apiUrl = `https://generativelanguage.googleapis.com/v1beta/models/gemini-2.0-flash:generateContent?key=${apiKey}`;

            const response = await fetch(apiUrl, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(payload)
            });

            const result = await response.json();

            if (result.candidates && result.candidates.length > 0 &&
                result.candidates[0].content && result.candidates[0].content.parts &&
                result.candidates[0].content.parts.length > 0) {
                const text = result.candidates[0].content.parts[0].text;
                setLlmModalContent(text);
            } else {
                setLlmModalContent('Failed to generate insights. Please try again.');
                console.error('Gemini API response structure unexpected:', result);
            }
        } catch (error) {
            setLlmModalContent('Error generating insights: ' + error.message);
            console.error('Error calling Gemini API for insights:', error);
        } finally {
            setLlmLoading(false);
        }
    };

    // --- Start of manually converted JSX to React.createElement for demonstration ---
    const headerElement = React.createElement(
        'header',
        { className: 'w-full max-w-6xl bg-white shadow-md rounded-lg p-4 mb-6 flex flex-col sm:flex-row justify-between items-center' },
        React.createElement(
            'div',
            { className: 'flex items-center mb-4 sm:mb-0' },
            React.createElement(
                'svg',
                { xmlns: 'http://www.w3.org/2000/svg', width: '24', height: '24', viewBox: '0 0 24 24', fill: 'none', stroke: 'currentColor', strokeWidth: '2', strokeLinecap: 'round', strokeLinejoin: 'round', className: 'lucide lucide-instagram text-purple-600 mr-2' },
                React.createElement('rect', { width: '20', height: '20', x: '2', y: '2', rx: '5', ry: '5' }),
                React.createElement('path', { d: 'M16 11.37A4 4 0 1 1 12.63 8 4 4 0 0 1 16 11.37z' }),
                React.createElement('line', { x1: '17.5', x2: '17.51', y1: '6.5', y2: '6.5' })
            ),
            React.createElement('h1', { className: 'text-xl sm:text-2xl font-bold text-gray-800' }, 'Instagram Followers Scraper')
        ),
        React.createElement(
            'nav',
            { className: 'flex flex-wrap justify-center sm:justify-end items-center space-x-4' },
            React.createElement('button', { className: 'text-gray-600 hover:text-purple-600 font-medium transition duration-200 py-1' }, 'FAQ'),
            React.createElement('button', { className: 'bg-gradient-to-r from-purple-500 to-pink-500 text-white px-4 py-2 rounded-full shadow-lg hover:shadow-xl transition duration-300 transform hover:scale-105 font-semibold' }, 'UPGRADE TO PRO'),
            React.createElement(
                'button',
                { className: 'text-gray-600 hover:text-purple-600 transition duration-200' },
                React.createElement(
                    'svg',
                    { xmlns: 'http://www.w3.org/2000/svg', width: '24', height: '24', viewBox: '0 0 24 24', fill: 'none', stroke: 'currentColor', strokeWidth: '2', strokeLinecap: 'round', strokeLinejoin: 'round', className: 'lucide lucide-user' },
                    React.createElement('path', { d: 'M19 21v-2a4 4 0 0 0-4-4H9a4 4 0 0 0-4 4v2' }),
                    React.createElement('circle', { cx: '12', cy: '7', r: '4' })
                )
            )
        )
    );
    // --- End of manually converted JSX to React.createElement for demonstration ---


    return (
        React.createElement(
            'div',
            { className: 'min-h-screen bg-gray-100 flex flex-col items-center p-4 sm:p-6 lg:p-8 font-inter' },
            headerElement, // Render the manually created header element

            // The rest of the application still uses JSX for brevity in this example.
            // REMEMBER: ALL of this JSX MUST be transpiled by a build tool for a real Chrome Extension.
            React.createElement(
                'main',
                { className: 'w-full max-w-6xl grid grid-cols-1 md:grid-cols-2 gap-6' },
                /* Section 2: Scraping Status */
                React.createElement(
                    'section',
                    { className: 'bg-white shadow-md rounded-lg p-6' },
                    React.createElement(
                        'h2',
                        { className: 'text-2xl font-semibold text-gray-800 mb-4 flex items-center' },
                        React.createElement(
                            'svg',
                            { xmlns: 'http://www.w3.org/2000/svg', width: '24', height: '24', viewBox: '0 0 24 24', fill: 'none', stroke: 'currentColor', strokeWidth: '2', strokeLinecap: 'round', strokeLinejoin: 'round', className: 'lucide lucide-activity text-blue-500 mr-2' },
                            React.createElement('path', { d: 'M22 12h-4l-3 9L9 3l-3 9H2' })
                        ),
                        'Scraping Status'
                    ),
                    React.createElement(
                        'div',
                        { className: 'space-y-4' },
                        React.createElement(
                            'div',
                            null,
                            React.createElement('p', { className: 'text-gray-600 text-sm' }, 'Username:'),
                            React.createElement('p', { className: 'font-semibold text-lg text-gray-900' }, username)
                        ),
                        React.createElement(
                            'div',
                            null,
                            React.createElement('p', { className: 'text-gray-600 text-sm' }, 'Instagram Account:'),
                            React.createElement(
                                'a',
                                { href: instagramLink, target: '_blank', rel: 'noopener noreferrer', className: 'text-blue-500 hover:underline text-lg' },
                                instagramLink
                            )
                        ),
                        React.createElement(
                            'div',
                            null,
                            React.createElement('p', { className: 'text-gray-600 text-sm' }, 'Total Follower Count:'),
                            React.createElement('p', { className: 'font-semibold text-lg text-gray-900' }, followerCount.toLocaleString())
                        ),
                        React.createElement(
                            'div',
                            null,
                            React.createElement('p', { className: 'text-gray-600 text-sm mb-1' }, 'Followers Scraped:'),
                            React.createElement(
                                'div',
                                { className: 'w-full bg-gray-200 rounded-full h-4' },
                                React.createElement('div', {
                                    className: 'bg-green-500 h-4 rounded-full transition-all duration-500 ease-out',
                                    style: { width: `${progressPercentage}%` }
                                })
                            ),
                            React.createElement(
                                'p',
                                { className: 'text-sm text-gray-700 mt-1' },
                                scrapedCount.toLocaleString(),
                                ' / ',
                                followerCount.toLocaleString(),
                                ' (',
                                progressPercentage.toFixed(2),
                                '%)'
                            )
                        ),
                        React.createElement(
                            'div',
                            { className: 'flex flex-col sm:flex-row gap-4 mt-6' },
                            React.createElement(
                                'button',
                                {
                                    onClick: handleDownload,
                                    className: 'flex-1 bg-blue-600 text-white px-6 py-3 rounded-lg shadow-md hover:bg-blue-700 transition duration-200 font-semibold flex items-center justify-center'
                                },
                                React.createElement(
                                    'svg',
                                    { xmlns: 'http://www.w3.org/2000/svg', width: '20', height: '20', viewBox: '0 0 24 24', fill: 'none', stroke: 'currentColor', strokeWidth: '2', strokeLinecap: 'round', strokeLinejoin: 'round', className: 'lucide lucide-download mr-2' },
                                    React.createElement('path', { d: 'M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4' }),
                                    React.createElement('polyline', { points: '7 10 12 15 17 10' }),
                                    React.createElement('line', { x1: '12', x2: '12', y1: '15', y2: '3' })
                                ),
                                'Download Data'
                            ),
                            isScraping ? (
                                React.createElement(
                                    'button',
                                    {
                                        onClick: handleStopScraping,
                                        className: 'flex-1 bg-red-600 text-white px-6 py-3 rounded-lg shadow-md hover:bg-red-700 transition duration-200 font-semibold flex items-center justify-center'
                                    },
                                    React.createElement(
                                        'svg',
                                        { xmlns: 'http://www.w3.org/2000/svg', width: '20', height: '20', viewBox: '0 0 24 24', fill: 'none', stroke: 'currentColor', strokeWidth: '2', strokeLinecap: 'round', strokeLinejoin: 'round', className: 'lucide lucide-square mr-2' },
                                        React.createElement('rect', { width: '18', height: '18', x: '3', y: '3', rx: '2', ry: '2' })
                                    ),
                                    'Stop Scraping'
                                )
                            ) : (
                                React.createElement(
                                    'button',
                                    {
                                        onClick: handleStartScraping,
                                        className: 'flex-1 bg-green-600 text-white px-6 py-3 rounded-lg shadow-md hover:bg-green-700 transition duration-200 font-semibold flex items-center justify-center'
                                    },
                                    React.createElement(
                                        'svg',
                                        { xmlns: 'http://www.w3.org/2000/svg', width: '20', height: '20', viewBox: '0 0 24 24', fill: 'none', stroke: 'currentColor', strokeWidth: '2', strokeLinecap: 'round', strokeLinejoin: 'round', className: 'lucide lucide-play mr-2' },
                                        React.createElement('polygon', { points: '5 3 19 12 5 21 5 3' })
                                    ),
                                    'Start Scraping'
                                )
                            )
                        ),
                        React.createElement(
                            'button',
                            {
                                onClick: handleGenerateEngagementInsights,
                                className: 'w-full bg-indigo-600 text-white px-6 py-3 rounded-lg shadow-md hover:bg-indigo-700 transition duration-200 font-semibold flex items-center justify-center mt-4',
                                disabled: llmLoading
                            },
                            llmLoading ? (
                                React.createElement(
                                    'span',
                                    { className: 'flex items-center' },
                                    React.createElement(
                                        'svg',
                                        { className: 'animate-spin -ml-1 mr-3 h-5 w-5 text-white', xmlns: 'http://www.w3.org/2000/svg', fill: 'none', viewBox: '0 0 24 24' },
                                        React.createElement('circle', { className: 'opacity-25', cx: '12', cy: '12', r: '10', stroke: 'currentColor', strokeWidth: '4' }),
                                        React.createElement('path', { className: 'opacity-75', fill: 'currentColor', d: 'M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z' })
                                    ),
                                    'Generating...'
                                )
                            ) : (
                                React.createElement(
                                    React.Fragment,
                                    null,
                                    React.createElement(
                                        'svg',
                                        { xmlns: 'http://www.w3.org/2000/svg', width: '20', height: '20', viewBox: '0 0 24 24', fill: 'none', stroke: 'currentColor', strokeWidth: '2', strokeLinecap: 'round', strokeLinejoin: 'round', className: 'lucide lucide-lightbulb mr-2' },
                                        React.createElement('path', { d: 'M15 14c.2-1 .7-1.7 1.5-2.5 1-.9 1.5-2.2 1.5-3.5A6 6 0 0 0 6 8c0 1.3.5 2.6 1.5 3.5.8.7 1.3 1.5 1.5 2.5' }),
                                        React.createElement('path', { d: 'M9 18h6' }),
                                        React.createElement('path', { d: 'M10 22h4' })
                                    ),
                                    'Generate Engagement Insights ✨'
                                )
                            )
                        )
                    )
                ),

                /* Section 3: Premium Feature Ad */
                React.createElement(
                    'section',
                    { className: 'bg-gradient-to-br from-purple-600 to-pink-600 text-white shadow-md rounded-lg p-6 flex flex-col justify-center items-center text-center' },
                    React.createElement('h2', { className: 'text-2xl font-bold mb-3' }, 'Unlock More Power with Premium!'),
                    React.createElement(
                        'p',
                        { className: 'text-lg mb-4 opacity-90' },
                        'Get advanced filters, unlimited exports, and real-time insights.'
                    ),
                    React.createElement(
                        'button',
                        { className: 'bg-white text-purple-600 px-6 py-3 rounded-full shadow-lg hover:shadow-xl transition duration-300 transform hover:scale-105 font-bold text-lg' },
                        'Learn More'
                    ),
                    React.createElement('p', { className: 'text-sm mt-3 opacity-80' }, 'Starting at just $9.99/month')
                ),

                /* Section 4: Scraped Users Table */
                React.createElement(
                    'section',
                    { className: 'md:col-span-2 bg-white shadow-md rounded-lg p-6' },
                    React.createElement(
                        'h2',
                        { className: 'text-2xl font-semibold text-gray-800 mb-4 flex items-center' },
                        React.createElement(
                            'svg',
                            { xmlns: 'http://www.w3.org/2000/svg', width: '24', height: '24', viewBox: '0 0 24 24', fill: 'none', stroke: 'currentColor', strokeWidth: '2', strokeLinecap: 'round', strokeLinejoin: 'round', className: 'lucide lucide-users text-green-500 mr-2' },
                            React.createElement('path', { d: 'M16 21v-2a4 4 0 0 0-4-4H6a4 4 0 0 0-4 4v2' }),
                            React.createElement('circle', { cx: '9', cy: '7', r: '4' }),
                            React.createElement('path', { d: 'M22 21v-2a4 4 0 0 0-3-3.87c-.5-.06-.5-1.12-.5-1.12A4 4 0 0 0 16 12a4 4 0 0 0-4-4h-1' })
                        ),
                        'Scraped Users'
                    ),
                    React.createElement(
                        'div',
                        { className: 'overflow-x-auto rounded-lg border border-gray-200' },
                        React.createElement(
                            'table',
                            { className: 'min-w-full divide-y divide-gray-200' },
                            React.createElement(
                                'thead',
                                { className: 'bg-gray-50' },
                                React.createElement(
                                    'tr',
                                    null,
                                    React.createElement('th', { scope: 'col', className: 'px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider' }, 'NO'),
                                    React.createElement('th', { scope: 'col', className: 'px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider' }, 'Avatar'),
                                    React.createElement('th', { scope: 'col', className: 'px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider' }, 'Username'),
                                    React.createElement('th', { scope: 'col', className: 'px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider' }, 'Full Name'),
                                    React.createElement('th', { scope: 'col', className: 'px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider' }, 'User ID'),
                                    React.createElement('th', { scope: 'col', className: 'px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider' }, 'Followed by You'),
                                    React.createElement('th', { scope: 'col', className: 'px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider' }, 'Verified'),
                                    React.createElement('th', { scope: 'col', className: 'px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider' }, 'Actions')
                                )
                            ),
                            React.createElement(
                                'tbody',
                                { className: 'bg-white divide-y divide-gray-200' },
                                scrapedUsers.map((user) =>
                                    React.createElement(
                                        'tr',
                                        { key: user.userId },
                                        React.createElement('td', { className: 'px-6 py-4 whitespace-nowrap text-sm text-gray-900' }, user.no),
                                        React.createElement(
                                            'td',
                                            { className: 'px-6 py-4 whitespace-nowrap' },
                                            React.createElement('img', {
                                                className: 'h-10 w-10 rounded-full',
                                                src: user.avatar,
                                                alt: `${user.username} avatar`,
                                                onError: (e) => {
                                                    e.target.onerror = null;
                                                    e.target.src = `https://placehold.co/40x40/cccccc/000000?text=${user.username.charAt(0).toUpperCase()}`;
                                                }
                                            })
                                        ),
                                        React.createElement('td', { className: 'px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900' }, user.username),
                                        React.createElement('td', { className: 'px-6 py-4 whitespace-nowrap text-sm text-gray-500' }, user.fullname),
                                        React.createElement('td', { className: 'px-6 py-4 whitespace-nowrap text-sm text-gray-500' }, user.userId),
                                        React.createElement(
                                            'td',
                                            { className: 'px-6 py-4 whitespace-nowrap text-sm' },
                                            user.followedByYou ? (
                                                React.createElement(
                                                    'span',
                                                    { className: 'px-2 inline-flex text-xs leading-5 font-semibold rounded-full bg-green-100 text-green-800' },
                                                    'Yes'
                                                )
                                            ) : (
                                                React.createElement(
                                                    'span',
                                                    { className: 'px-2 inline-flex text-xs leading-5 font-semibold rounded-full bg-red-100 text-red-800' },
                                                    'No'
                                                )
                                            )
                                        ),
                                        React.createElement(
                                            'td',
                                            { className: 'px-6 py-4 whitespace-nowrap text-sm' },
                                            user.isVerified ? (
                                                React.createElement(
                                                    'svg',
                                                    { xmlns: 'http://www.w3.org/2000/svg', width: '20', height: '20', viewBox: '0 0 24 24', fill: 'none', stroke: 'currentColor', strokeWidth: '2', strokeLinecap: 'round', strokeLinejoin: 'round', className: 'lucide lucide-badge-check text-blue-500' },
                                                    React.createElement('path', { d: 'M3.85 8.62a4 4 0 0 1 4.78-4.77 4 4 0 0 1 6.74 0 4 4 0 0 1 4.78 4.78 4 4 0 0 1 0 6.74 4 4 0 0 1-4.77 4.78 4 4 0 0 1-6.75 0 4 4 0 0 1-4.78-4.77 4 4 0 0 1 0-6.76Z' }),
                                                    React.createElement('path', { d: 'm9 12 2 2 4-4' })
                                                )
                                            ) : (
                                                React.createElement(
                                                    'svg',
                                                    { xmlns: 'http://www.w3.org/2000/svg', width: '20', height: '20', viewBox: '0 0 24 24', fill: 'none', stroke: 'currentColor', strokeWidth: '2', strokeLinecap: 'round', strokeLinejoin: 'round', className: 'lucide lucide-x-circle text-gray-400' },
                                                    React.createElement('circle', { cx: '12', cy: '12', r: '10' }),
                                                    React.createElement('path', { d: 'm15 9-6 6' }),
                                                    React.createElement('path', { d: 'm9 9 6 6' })
                                                )
                                            )
                                        ),
                                        React.createElement(
                                            'td',
                                            { className: 'px-6 py-4 whitespace-nowrap text-right text-sm font-medium' },
                                            React.createElement(
                                                'button',
                                                {
                                                    onClick: () => handleGenerateBioIdea(user),
                                                    className: 'text-indigo-600 hover:text-indigo-900 bg-indigo-100 hover:bg-indigo-200 px-3 py-1 rounded-md transition duration-150 ease-in-out font-semibold',
                                                    disabled: llmLoading
                                                },
                                                llmLoading ? 'Generating...' : 'Generate Bio Idea ✨'
                                            )
                                        )
                                    )
                                )
                            )
                        )
                    )
                ),

                /* LLM Output Modal */
                showLLMModal &&
                React.createElement(
                    'div',
                    { className: 'fixed inset-0 bg-gray-600 bg-opacity-50 flex items-center justify-center z-50 p-4' },
                    React.createElement(
                        'div',
                        { className: 'bg-white p-6 rounded-lg shadow-xl text-center max-w-lg w-full' },
                        React.createElement('h3', { className: 'text-xl font-bold mb-4 text-gray-800' }, llmModalTitle),
                        llmLoading ? (
                            React.createElement(
                                'div',
                                { className: 'flex justify-center items-center h-24' },
                                React.createElement(
                                    'svg',
                                    { className: 'animate-spin h-8 w-8 text-blue-500', xmlns: 'http://www.w3.org/2000/svg', fill: 'none', viewBox: '0 0 24 24' },
                                    React.createElement('circle', { className: 'opacity-25', cx: '12', cy: '12', r: '10', stroke: 'currentColor', strokeWidth: '4' }),
                                    React.createElement('path', { className: 'opacity-75', fill: 'currentColor', d: 'M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z' })
                                ),
                                React.createElement('p', { className: 'ml-3 text-gray-700' }, 'Loading...')
                            )
                        ) : (
                            React.createElement('p', { className: 'text-gray-700 whitespace-pre-wrap text-left' }, llmModalContent)
                        ),
                        React.createElement(
                            'button',
                            {
                                onClick: () => setShowLLMModal(false),
                                className: 'mt-6 bg-blue-600 text-white px-5 py-2 rounded-lg hover:bg-blue-700 transition duration-200 font-semibold'
                            },
                            'Close'
                        )
                    )
                )
            )
        )
    );
};

// ReactDOM.render for embedding React into HTML
const rootElement = document.getElementById('root');
ReactDOM.createRoot(rootElement).render(React.createElement(App, null));

// Ensure the font is loaded
// For a Chrome extension, you should ideally bundle the font files locally
const link = document.createElement('link');
link.href = 'https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap';
link.rel = 'stylesheet';
document.head.appendChild(link);
